<?php
defined('BASEPATH') OR exit('No direct script access allowed');

// Manually load Frontend_Controller if not autoloaded
if (!class_exists('Frontend_Controller')) {
    require_once APPPATH . 'core/Frontend_Controller.php';
}

/**
 * Blog Controller
 * Handles blog listing, single post, and category pages
 * Location: application/controllers/Blog.php
 */

class Blog extends Frontend_Controller {

    public function __construct() {
        parent::__construct();
        $this->load->library('pagination');
    }

    /**
     * Blog listing page
     */
    public function index($page = 0) {
        // Pagination configuration
        $per_page = 9;
        $total_rows = $this->Blog_model->count_posts('published');
        
        $config = generate_pagination(base_url('blog/page'), $total_rows, $per_page, 3);
        $this->pagination->initialize($config);

        // Get blog posts
        $data['posts'] = $this->Blog_model->get_posts($per_page, $page, 'published');
        $data['pagination'] = $this->pagination->create_links();

        // Set meta tags
        $this->set_meta(
            'Blog',
            'Read our latest news, stories, and updates from the community',
            'blog, news, stories, community updates, ngo news'
        );

        // Set page heading
        $this->set_heading('Our Blog', 'Latest News & Stories');

        // Breadcrumbs
        $this->add_breadcrumb('Home', base_url());
        $this->add_breadcrumb('Blog', null);

        // Render view
        $this->render('blog/index', $data);
    }

    /**
     * Single blog post
     */
    public function view($slug) {
        // Get post by slug
        $post = $this->Blog_model->get_post_by_slug($slug);

        // Check if post exists
        if (!$post) {
            show_404();
        }

        // Increment view count
        $this->Blog_model->increment_views($post->id);

        // Get related posts
        $data['post'] = $post;
        $data['related_posts'] = $this->Blog_model->get_related_posts($post->category_id, $post->id, 3);

        // Set meta tags
        $meta_title = !empty($post->meta_title) ? $post->meta_title : $post->title;
        $meta_description = !empty($post->meta_description) ? $post->meta_description : $post->excerpt;
        $meta_image = !empty($post->featured_image) ? base_url('uploads/blog/' . $post->featured_image) : null;

        $this->set_meta($meta_title, $meta_description, null, $meta_image);

        // Set page heading
        $this->set_heading($post->title, format_date($post->published_at));

        // Breadcrumbs
        $this->add_breadcrumb('Home', base_url());
        $this->add_breadcrumb('Blog', base_url('blog'));
        if ($post->category_name) {
            $this->add_breadcrumb($post->category_name, base_url('blog/category/' . $post->category_slug));
        }
        $this->add_breadcrumb($post->title, null);

        // Render view
        $this->render('blog/single', $data);
    }

    /**
     * Blog category page
     */
    public function category($category_slug, $page = 0) {
        // Get category
        $category = $this->Blog_model->get_category_by_slug($category_slug);

        if (!$category) {
            show_404();
        }

        // Pagination configuration
        $per_page = 9;
        $total_rows = $this->Blog_model->count_posts('published', $category->id);
        
        $config = generate_pagination(base_url('blog/category/' . $category_slug . '/page'), $total_rows, $per_page, 4);
        $this->pagination->initialize($config);

        // Get posts by category
        $data['posts'] = $this->Blog_model->get_posts_by_category($category_slug, $per_page, $page);
        $data['category'] = $category;
        $data['pagination'] = $this->pagination->create_links();

        // Set meta tags
        $this->set_meta(
            $category->name . ' - Blog',
            $category->description ?: 'Read articles in ' . $category->name . ' category'
        );

        // Set page heading
        $this->set_heading($category->name, $category->description);

        // Breadcrumbs
        $this->add_breadcrumb('Home', base_url());
        $this->add_breadcrumb('Blog', base_url('blog'));
        $this->add_breadcrumb($category->name, null);

        // Render view
        $this->render('blog/category', $data);
    }
}