<?php
defined('BASEPATH') OR exit('No direct script access allowed');

// Manually load Frontend_Controller if not autoloaded
if (!class_exists('Frontend_Controller')) {
    require_once APPPATH . 'core/Frontend_Controller.php';
}

/**
 * Consultation Controller
 * Handles free 1-on-1 consultation registration
 * Location: application/controllers/Consultation.php
 */
class Consultation extends Frontend_Controller {
    
    public function __construct() {
        parent::__construct();
        $this->load->model('Consultation_model');
        $this->load->library('form_validation');
    }

    /**
     * Consultation registration form
     */
    public function index() {
        // Set meta tags
        $this->set_meta(
            'Register for Free Consultation',
            'Book a free 1-on-1 consultation session with our experts. We are here to help you achieve your goals.',
            'consultation, free consultation, 1-on-1 session, expert advice'
        );

        // Set page heading
        $this->set_heading('Register With Us', 'Book Your Free 1-on-1 Consultation');

        // Breadcrumbs
        $this->add_breadcrumb('Home', base_url());
        $this->add_breadcrumb('Register for Consultation', null);

        // Render view
        $this->render('consultation/form');
    }

    /**
     * Submit consultation request
     */
    public function submit() {
        // Check if form is submitted via POST
        if ($this->input->method() !== 'post') {
            redirect('consultation');
            return;
        }

        log_message('info', '=== CONSULTATION FORM SUBMISSION START ===');

        // Set validation rules
        $this->form_validation->set_rules('full_name', 'Full Name', 'required|max_length[255]');
        $this->form_validation->set_rules('email', 'Email', 'required|valid_email|max_length[255]');
        $this->form_validation->set_rules('phone', 'Phone Number', 'required|max_length[20]');
        $this->form_validation->set_rules('organization', 'Organization', 'max_length[255]');
        $this->form_validation->set_rules('consultation_type', 'Consultation Type', 'required|in_list[general,program,volunteer,partnership,other]');
       $this->form_validation->set_rules('preferred_date', 'Preferred Date', 'trim');
        $this->form_validation->set_rules('preferred_time', 'Preferred Time', 'in_list[morning,afternoon,evening]');
        $this->form_validation->set_rules('subject', 'Subject', 'required|max_length[255]');
        $this->form_validation->set_rules('message', 'Message', 'required|max_length[5000]');

        // Run validation
        if ($this->form_validation->run() === FALSE) {
            log_message('error', 'Validation failed: ' . validation_errors());
            // Validation failed - redisplay form with errors
            $this->index();
            return;
        }

        log_message('info', 'Validation passed');

        // Get form data
        $email = $this->input->post('email', TRUE);
        $full_name = $this->input->post('full_name', TRUE);

        log_message('info', 'Processing consultation for: ' . $email);

        // Check if email already has pending consultation
        if ($this->Consultation_model->has_pending_consultation($email)) {
            log_message('warning', 'Duplicate pending consultation for: ' . $email);
            $this->session->set_flashdata('warning', 'You already have a pending consultation request. We will contact you soon.');
            redirect('consultation');
            return;
        }

        // Prepare data array
        $data = array(
            'full_name' => $full_name,
            'email' => $email,
            'phone' => $this->input->post('phone', TRUE),
            'organization' => $this->input->post('organization', TRUE),
            'consultation_type' => $this->input->post('consultation_type', TRUE),
            'preferred_date' => $this->input->post('preferred_date', TRUE) ?: NULL,
            'preferred_time' => $this->input->post('preferred_time', TRUE) ?: NULL,
            'subject' => $this->input->post('subject', TRUE),
            'message' => $this->input->post('message', TRUE),
            'status' => 'pending'
        );

        log_message('info', 'Data prepared: ' . json_encode($data));

        // Save to database
        log_message('info', 'Attempting to save consultation to database');
        $consultation_id = $this->Consultation_model->create_consultation($data);

        if ($consultation_id) {
            log_message('info', 'Consultation created successfully with ID: ' . $consultation_id);

            // Set success message BEFORE any redirects
            $this->session->set_flashdata('success', 'Thank you! Your consultation request has been submitted successfully. We will contact you within 24-48 hours.');

            try {
                // Get consultation data
                $consultation = $this->Consultation_model->get_consultation($consultation_id);

                // Send confirmation email to user
                if (function_exists('send_consultation_confirmation') && $consultation) {
                    send_consultation_confirmation($consultation);
                    log_message('info', 'Confirmation email sent to: ' . $email);
                }

                // Send notification email to admin
                if (function_exists('send_consultation_notification_to_admin') && $consultation) {
                    send_consultation_notification_to_admin($consultation);
                    log_message('info', 'Admin notification email sent');
                }
            } catch (Exception $e) {
                log_message('error', 'Error sending emails: ' . $e->getMessage());
                // Don't fail the submission if emails fail
            }

            log_message('info', '=== CONSULTATION FORM SUBMISSION SUCCESS ===');
            redirect('consultation');
        } else {
            log_message('error', 'Failed to create consultation for email: ' . $email);
            $this->session->set_flashdata('error', 'An error occurred while submitting your request. Please try again.');
            redirect('consultation');
        }
    }
	public function validate_date($date)
{
    if (empty($date)) return TRUE; // optional field

    $d = DateTime::createFromFormat('Y-m-d', $date);
    return $d && $d->format('Y-m-d') === $date;
}

}