<?php
defined('BASEPATH') OR exit('No direct script access allowed');

// Manually load Frontend_Controller if not autoloaded
if (!class_exists('Frontend_Controller')) {
    require_once APPPATH . 'core/Frontend_Controller.php';
}

class Kid_registration extends Frontend_Controller {
    public function __construct() {
        parent::__construct();
        $this->load->model('Kid_registration_model');
        $this->load->helper('form');
        $this->load->library('form_validation');
    }

    public function index() {
        // Set meta tags
        $this->set_meta(
            'Kid French Learner Registration', 
            'Register for French language classes for children', 
            'french learning, kid registration, children language course'
        );

        // Set page heading
        $this->set_heading('French Language Registration', 'Kid Learner Form');

        // Add breadcrumbs
        $this->add_breadcrumb('Home', base_url());
        $this->add_breadcrumb('Kid French Registration', null);

        // Prepare data for the view
        $data = [
            'title' => 'Kid French Learner Registration'
        ];

        // Render the view using the custom render method
        $this->render('french_registration/kid_form', $data);
    }

    public function submit() {
        // Explicitly check for POST method
        if ($this->input->method(TRUE) !== 'POST') {
            show_error('Invalid request method');
        }

        // Set validation rules
        $this->form_validation->set_rules('first_name', 'First Name', 'required|trim|max_length[100]');
        $this->form_validation->set_rules('surname', 'Surname', 'required|trim|max_length[100]');
        $this->form_validation->set_rules('age', 'Age', 'required|numeric|greater_than[0]|less_than[18]');
        $this->form_validation->set_rules('gender', 'Gender', 'required|in_list[Male,Female,Other]');
        $this->form_validation->set_rules('parent_name', 'Parent Name', 'required|trim|max_length[200]');
        $this->form_validation->set_rules('email', 'Email', 'required|valid_email|is_unique[french_kid_learners.email]');
        $this->form_validation->set_rules('contact_number', 'Contact Number', 'required|trim');
        $this->form_validation->set_rules('home_location', 'Home Location', 'required|trim');
        $this->form_validation->set_rules('french_level', 'French Level', 'required');

        // Log the submission attempt
        log_message('info', 'Kid French Registration Submission Attempt');

        if ($this->form_validation->run() == FALSE) {
            // Log validation errors
            log_message('error', 'Validation failed: ' . validation_errors());
            
            // If validation fails, reload the form with error messages
            $this->index();
            return;
        }

        // Prepare registration data
        $registration_data = [
            'first_name' => $this->input->post('first_name'),
            'surname' => $this->input->post('surname'),
            'age' => $this->input->post('age'),
            'gender' => $this->input->post('gender'),
            'parent_name' => $this->input->post('parent_name'),
            'email' => $this->input->post('email'),
            'contact_number' => $this->input->post('contact_number'),
            'home_location' => $this->input->post('home_location'),
            'french_level' => $this->input->post('french_level'),
            'current_school' => $this->input->post('current_school'),
            'current_grade' => $this->input->post('current_grade'),
            'favorite_french_activity' => $this->input->post('favorite_french_activity'),
            'area_of_improvement' => $this->input->post('area_of_improvement'),
            'learning_behavioral_difficulties' => $this->input->post('learning_behavioral_difficulties'),
            'allergies' => $this->input->post('allergies')
        ];

        // Log registration data (be careful with sensitive information)
        log_message('info', 'Registration data prepared for: ' . $registration_data['email']);

        // Attempt to register
        try {
            $result = $this->Kid_registration_model->register($registration_data);

            if ($result) {
                // Registration successful
                log_message('info', 'Kid registration successful for: ' . $registration_data['email']);
                
                // Try to send confirmation email (if function exists)
                if (function_exists('send_registration_confirmation')) {
                    try {
                        send_registration_confirmation($registration_data);
                    } catch (Exception $e) {
                        log_message('error', 'Failed to send confirmation email: ' . $e->getMessage());
                    }
                }

                $this->session->set_flashdata('success', 'Your child\'s registration has been submitted successfully!');
                redirect('french-registration/kid');
            } else {
                // Registration failed
                log_message('error', 'Kid registration failed for: ' . $registration_data['email']);
                $this->session->set_flashdata('error', 'Registration failed. Please try again.');
                $this->index();
            }
        } catch (Exception $e) {
            // Log any unexpected errors
            log_message('error', 'Unexpected error in kid registration: ' . $e->getMessage());
            $this->session->set_flashdata('error', 'An unexpected error occurred. Please try again later.');
            $this->index();
        }
    }
}