<?php
defined('BASEPATH') OR exit('No direct script access allowed');

// Manually load Frontend_Controller if not autoloaded
if (!class_exists('Frontend_Controller')) {
    require_once APPPATH . 'core/Frontend_Controller.php';
}

/**
 * Volunteers Controller
 * Handles volunteer listing and application
 * Location: application/controllers/Volunteers.php
 */

class Volunteers extends Frontend_Controller {

    public function __construct() {
        parent::__construct();
        $this->load->model('Volunteer_model');
        $this->load->library('form_validation');
		
		// Make sure upload library is available
    if (!isset($this->upload)) {
        $this->load->library('upload');
    }
    }

    /**
     * Our volunteers page - display active volunteers
     */
    public function index() {
        // Get active volunteers
        $data['volunteers'] = $this->Volunteer_model->get_active_volunteers();

        // Set meta tags
        $this->set_meta(
            'Our Volunteers',
            'Meet our dedicated volunteers who are making a difference in the community',
            'volunteers, community volunteers, volunteer team'
        );

        // Set page heading
        $this->set_heading('Our Volunteers', 'The Heart of Our Organization');

        // Breadcrumbs
        $this->add_breadcrumb('Home', base_url());
        $this->add_breadcrumb('Our Volunteers', null);

        // Render view
        $this->render('volunteers/index', $data);
    }

    /**
     * Volunteer application form
     */
    public function apply() {
        // Set meta tags
        $this->set_meta(
            'Volunteer With Us',
            'Join our volunteer team and make a positive impact in your community. Apply now!',
            'volunteer application, join volunteers, become volunteer'
        );

        // Set page heading
        $this->set_heading('Volunteer With Us', 'Join Our Team and Make a Difference');

        // Breadcrumbs
        $this->add_breadcrumb('Home', base_url());
        $this->add_breadcrumb('Volunteer Application', null);

        // Render view
        $this->render('volunteers/application_form');
    }

    /**
     * Submit volunteer application
     */
public function submit() {
    // Check if form is submitted via POST
    if ($this->input->method() !== 'post') {
        redirect('volunteer/apply');
        return;
    }

    log_message('info', '=== VOLUNTEER FORM SUBMISSION START ===');

    // Set validation rules
    $this->form_validation->set_rules('full_name', 'Full Name', 'required|max_length[255]');
    $this->form_validation->set_rules('email', 'Email', 'required|valid_email|max_length[255]');
    $this->form_validation->set_rules('phone', 'Phone', 'required|max_length[20]');
    $this->form_validation->set_rules('address', 'Address', 'required|max_length[500]');
    $this->form_validation->set_rules('city', 'City', 'required|max_length[100]');
    $this->form_validation->set_rules('country', 'Country', 'max_length[100]');
    $this->form_validation->set_rules('occupation', 'Occupation', 'required|max_length[100]');
    $this->form_validation->set_rules('skills', 'Skills', 'required|max_length[500]');
    $this->form_validation->set_rules('availability', 'Availability', 'required|in_list[weekdays,weekends,flexible]');
    $this->form_validation->set_rules('areas_of_interest', 'Areas of Interest', 'required|max_length[1000]');
    $this->form_validation->set_rules('motivation', 'Motivation', 'required|max_length[2000]');
    $this->form_validation->set_rules('emergency_contact_name', 'Emergency Contact Name', 'required|max_length[255]');
    $this->form_validation->set_rules('emergency_contact_phone', 'Emergency Contact Phone', 'required|max_length[20]');

    // Run validation
    if ($this->form_validation->run() === FALSE) {
        log_message('error', 'Validation failed: ' . validation_errors());
        $this->apply();
        return;
    }

    log_message('info', 'Validation passed');

    // Check if email already exists
    $email = $this->input->post('email', TRUE);
    log_message('info', 'Checking email: ' . $email);

    if ($this->Volunteer_model->email_exists($email)) {
        log_message('warning', 'Email already exists: ' . $email);
        $this->session->set_flashdata('warning', 'This email is already registered. If you need to update your information, please contact us.');
        redirect('volunteer/apply');
        return;
    }

    // Prepare data array
    $data = array(
        'full_name' => $this->input->post('full_name', TRUE),
        'email' => $email,
        'phone' => $this->input->post('phone', TRUE),
        'address' => $this->input->post('address', TRUE),
        'city' => $this->input->post('city', TRUE),
        'country' => $this->input->post('country', TRUE) ?: 'Ghana',
        'date_of_birth' => $this->input->post('date_of_birth', TRUE) ?: NULL,
        'occupation' => $this->input->post('occupation', TRUE),
        'skills' => $this->input->post('skills', TRUE),
        'availability' => $this->input->post('availability', TRUE),
        'areas_of_interest' => $this->input->post('areas_of_interest', TRUE),
        'previous_experience' => $this->input->post('previous_experience', TRUE),
        'motivation' => $this->input->post('motivation', TRUE),
        'emergency_contact_name' => $this->input->post('emergency_contact_name', TRUE),
        'emergency_contact_phone' => $this->input->post('emergency_contact_phone', TRUE),
        'status' => 'pending'
    );

    log_message('info', 'Data prepared for insertion');

    // Handle profile photo upload - DIRECT METHOD
// Handle profile photo upload - MANUAL METHOD
if (!empty($_FILES['profile_photo']['name']) && $_FILES['profile_photo']['error'] === 0) {
    log_message('info', '========== MANUAL FILE UPLOAD START ==========');
    
    $file = $_FILES['profile_photo'];
    $upload_dir = FCPATH . 'uploads/volunteers/';
    
    // Create directory if doesn't exist
    if (!is_dir($upload_dir)) {
        mkdir($upload_dir, 0777, true);
    }
    
    // Get file extension
    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    
    // Check allowed extensions
    $allowed = array('jpg', 'jpeg', 'png', 'gif');
    
    if (in_array($ext, $allowed)) {
        // Generate unique filename
        $new_filename = uniqid() . '_' . time() . '.' . $ext;
        $destination = $upload_dir . $new_filename;
        
        log_message('info', 'Temp file: ' . $file['tmp_name']);
        log_message('info', 'Destination: ' . $destination);
        log_message('info', 'Temp file exists: ' . (file_exists($file['tmp_name']) ? 'YES' : 'NO'));
        
        // Move uploaded file
        if (move_uploaded_file($file['tmp_name'], $destination)) {
            $data['profile_photo'] = $new_filename;
            log_message('info', '✓ SUCCESS: File saved as: ' . $new_filename);
            log_message('info', 'File exists in destination: ' . (file_exists($destination) ? 'YES' : 'NO'));
        } else {
            log_message('error', '✗ FAILED: Could not move uploaded file');
            log_message('error', 'Error: ' . error_get_last()['message']);
        }
    } else {
        log_message('error', '✗ FAILED: File type not allowed: ' . $ext);
    }
    
    log_message('info', '========== MANUAL FILE UPLOAD END ==========');
}
  // Save to database
    log_message('info', 'Attempting to save volunteer application to database');
    $volunteer_id = $this->Volunteer_model->create_application($data);
    
    if ($volunteer_id) {
        log_message('info', 'Volunteer application created successfully with ID: ' . $volunteer_id);
        
        $this->session->set_flashdata('success', 'Thank you for your application! We will review it and get back to you within 5-7 business days.');
        
        // Get volunteer data
        $volunteer = $this->Volunteer_model->get_volunteer($volunteer_id);
        
        // Send emails
        if (function_exists('send_volunteer_confirmation') && $volunteer) {
            send_volunteer_confirmation($volunteer);
            log_message('info', 'Confirmation email sent to: ' . $email);
        }
        
        log_message('info', '=== VOLUNTEER FORM SUBMISSION SUCCESS ===');
        redirect('volunteer/apply');
    } else {
        log_message('error', 'Failed to create volunteer application');
        $this->session->set_flashdata('error', 'An error occurred while submitting your application. Please try again.');
        redirect('volunteer/apply');
    }
}
}