<?php
defined('BASEPATH') OR exit('No direct script access allowed');

// Manually load Admin_Controller if not autoloaded
if (!class_exists('Admin_Controller')) {
    require_once APPPATH . 'core/Admin_Controller.php';
}

/**
 * Admin Dashboard Controller
 * Handles main admin dashboard with statistics and overview
 * Location: application/controllers/admin/Dashboard.php
 */

class Dashboard extends Admin_Controller {

    public function __construct() {
        parent::__construct();
    }

    /**
     * Dashboard home
     */
    public function index() {
        $this->set_title('Dashboard', 'Overview & Statistics');

        // Get comprehensive statistics
        $data['stats'] = array(
            // Blog stats
            'total_posts' => $this->Blog_model->count_posts(),
            'published_posts' => $this->Blog_model->count_posts('published'),
            'draft_posts' => $this->Blog_model->count_posts('draft'),
            
            // Volunteer stats
            'total_volunteers' => $this->Volunteer_model->count_volunteers(),
            'pending_volunteers' => $this->Volunteer_model->count_volunteers('pending'),
            'active_volunteers' => $this->Volunteer_model->count_volunteers('active'),
            'approved_volunteers' => $this->Volunteer_model->count_volunteers('approved'),
            
            // Consultation stats
            'total_consultations' => $this->Consultation_model->count_consultations(),
            'new_consultations' => $this->Consultation_model->count_consultations('new'),
            'scheduled_consultations' => $this->Consultation_model->count_consultations('scheduled'),
            'completed_consultations' => $this->Consultation_model->count_consultations('completed'),
            
            // Message stats
            'total_messages' => $this->Contact_model->count_messages(),
            'unread_messages' => $this->Contact_model->count_messages('new'),
            'replied_messages' => $this->Contact_model->count_messages('replied'),
            
            // Other stats
            'total_workshops' => count($this->Common_model->get_workshops()),
            'upcoming_workshops' => count($this->Common_model->get_workshops('upcoming')),
            'active_partners' => count($this->Common_model->get_partners('active'))
        );

        // Recent activities
        $data['recent_consultations'] = $this->Consultation_model->get_consultations(5, 0);
        $data['recent_volunteers'] = $this->Volunteer_model->get_volunteers(5, 0);
        $data['recent_messages'] = $this->Contact_model->get_messages(5, 0);
        $data['upcoming_scheduled'] = $this->Consultation_model->get_upcoming_consultations(5);

        // Recent blog posts
        $data['recent_posts'] = $this->Blog_model->get_posts(5, 0);

        // Monthly statistics (last 6 months)
        $data['monthly_stats'] = $this->get_monthly_statistics();

        $this->render('dashboard/index', $data);
    }

    /**
     * Get monthly statistics for charts
     */
    private function get_monthly_statistics() {
        $stats = array();
        
        for ($i = 5; $i >= 0; $i--) {
            $month_start = date('Y-m-01', strtotime("-$i months"));
            $month_end = date('Y-m-t', strtotime("-$i months"));
            $month_name = date('M Y', strtotime("-$i months"));
            
            $stats[] = array(
                'month' => $month_name,
                'volunteers' => $this->get_monthly_count_range('volunteers', 'application_date', $month_start, $month_end),
                'consultations' => $this->get_monthly_count_range('consultation_requests', 'created_at', $month_start, $month_end),
                'messages' => $this->get_monthly_count_range('contact_messages', 'created_at', $month_start, $month_end),
                'blog_posts' => $this->get_monthly_count_range_published('blog_posts', 'created_at', $month_start, $month_end)
            );
        }
        
        return $stats;
    }

    /**
     * Get count for specific month range
     */
    private function get_monthly_count_range($table, $date_field, $start_date, $end_date) {
        try {
            $this->db->where("$date_field >=", $start_date);
            $this->db->where("$date_field <=", $end_date . ' 23:59:59');
            $this->db->where("$date_field IS NOT NULL");
            $count = $this->db->count_all_results($table);
            return (int)$count;
        } catch (Exception $e) {
            log_message('error', "Error getting count from $table: " . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get count for blog posts (only published)
     */
    private function get_monthly_count_range_published($table, $date_field, $start_date, $end_date) {
        try {
            $this->db->where("$date_field >=", $start_date);
            $this->db->where("$date_field <=", $end_date . ' 23:59:59');
            $this->db->where("$date_field IS NOT NULL");
            $this->db->where('status', 'published');
            $count = $this->db->count_all_results($table);
            return (int)$count;
        } catch (Exception $e) {
            log_message('error', "Error getting published posts count: " . $e->getMessage());
            return 0;
        }
    }

    /**
     * Quick actions (AJAX endpoints)
     */
    public function quick_action() {
        $action = $this->input->post('action');
        $id = $this->input->post('id');

        switch ($action) {
            case 'approve_volunteer':
                if ($this->Volunteer_model->approve_volunteer($id, $this->admin_id)) {
                    $volunteer = $this->Volunteer_model->get_volunteer($id);
                    send_volunteer_approval($volunteer);
                    $this->success_response('Volunteer approved successfully');
                } else {
                    $this->error_response('Failed to approve volunteer');
                }
                break;

            case 'mark_message_read':
                if ($this->Contact_model->mark_as_read($id)) {
                    $this->success_response('Message marked as read');
                } else {
                    $this->error_response('Failed to update message');
                }
                break;

            default:
                $this->error_response('Invalid action');
        }
    }
}