<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Frontend Base Controller
 * All frontend controllers should extend this class
 * Location: application/core/Frontend_Controller.php
 * 
 * Handles:
 * - Common data loading (settings, menus, etc.)
 * - Frontend layout rendering
 * - SEO meta tags
 * - Social sharing
 */

class Frontend_Controller extends CI_Controller {

    protected $data = array();

    public function __construct() {
        parent::__construct();
        
        // Load required libraries and helpers
        $this->load->helper(array('MY_utility', 'MY_email'));
        $this->load->model(array(
            'Common_model',
            'Blog_model'
        ));

        // Load common frontend data
        $this->load_common_data();
    }

    /**
     * Load common data for all frontend pages
     */
    protected function load_common_data() {
        // Site settings
        $this->data['site_settings'] = $this->Common_model->get_all_settings();
        
        // Default meta tags
        $this->data['meta_title'] = $this->data['site_settings']['site_name'] ?? 'NGO Platform';
        $this->data['meta_description'] = $this->data['site_settings']['about_description'] ?? 'Making a positive impact in our community';
        $this->data['meta_keywords'] = 'ngo, charity, volunteer, community, workshop, donation';
        $this->data['meta_image'] = base_url('assets/images/logo.png');
        
        // Social media links
        $this->data['social_links'] = array(
            'facebook' => $this->data['site_settings']['facebook_url'] ?? '#',
            'twitter' => $this->data['site_settings']['twitter_url'] ?? '#',
            'instagram' => $this->data['site_settings']['instagram_url'] ?? '#',
            'linkedin' => $this->data['site_settings']['linkedin_url'] ?? '#'
        );
        
        // Navigation menu (active class handling)
        $this->data['current_uri'] = uri_string();
        $this->data['current_segment'] = $this->uri->segment(1) ?: 'home';
        
        // Blog categories for menu/sidebar
        $this->data['blog_categories'] = $this->Blog_model->get_categories_with_count();
        
        // Recent blog posts for sidebar
        $this->data['recent_posts'] = $this->Blog_model->get_recent_posts(5);
        
        // Featured workshops
        $this->data['featured_workshops'] = $this->Common_model->get_featured_workshops(3);
        
        // Page title
        $this->data['page_title'] = '';
        $this->data['page_heading'] = '';
        $this->data['show_page_header'] = true;
        
        // Breadcrumbs
        $this->data['breadcrumbs'] = array();
    }

    /**
     * Render frontend view with layout
     */
    protected function render($view, $data = array(), $layout = 'main') {
        // Merge data
        $this->data = array_merge($this->data, $data);
        
        // Set full page title
        if (!empty($this->data['page_title'])) {
            $this->data['meta_title'] = $this->data['page_title'] . ' - ' . $this->data['site_settings']['site_name'];
        }
        
        // Load view parts
        $this->data['content_for_layout'] = $this->load->view('frontend/' . $view, $this->data, TRUE);
        $this->load->view('frontend/layouts/' . $layout, $this->data);
    }

    /**
     * Render view without layout (for AJAX)
     */
    protected function render_view($view, $data = array()) {
        $this->data = array_merge($this->data, $data);
        $this->load->view('frontend/' . $view, $this->data);
    }

    /**
     * Set page title and meta
     */
    protected function set_meta($title, $description = null, $keywords = null, $image = null) {
        $this->data['page_title'] = $title;
        $this->data['meta_title'] = $title . ' - ' . $this->data['site_settings']['site_name'];
        
        if ($description) {
            $this->data['meta_description'] = $description;
        }
        
        if ($keywords) {
            $this->data['meta_keywords'] = $keywords;
        }
        
        if ($image) {
            $this->data['meta_image'] = $image;
        }
    }

    /**
     * Set page heading
     */
    protected function set_heading($heading, $subheading = '') {
        $this->data['page_heading'] = $heading;
        $this->data['page_subheading'] = $subheading;
    }

    /**
     * Add breadcrumb
     */
    protected function add_breadcrumb($title, $url = null) {
        $this->data['breadcrumbs'][] = array(
            'title' => $title,
            'url' => $url
        );
    }

    /**
     * Disable page header
     */
    protected function disable_page_header() {
        $this->data['show_page_header'] = false;
    }

    /**
     * JSON response helper
     */
    protected function json_response($data) {
        $this->output
            ->set_content_type('application/json')
            ->set_output(json_encode($data));
    }

    /**
     * Success response
     */
    protected function success_response($message, $data = null) {
        $response = array(
            'status' => 'success',
            'message' => $message
        );
        
        if ($data) {
            $response['data'] = $data;
        }
        
        $this->json_response($response);
    }

    /**
     * Error response
     */
    protected function error_response($message, $errors = null) {
        $response = array(
            'status' => 'error',
            'message' => $message
        );
        
        if ($errors) {
            $response['errors'] = $errors;
        }
        
        $this->json_response($response);
    }

    /**
     * Validate form with specific rules
     */
    protected function validate_form($rules) {
        $this->form_validation->set_rules($rules);
        
        if ($this->form_validation->run() === FALSE) {
            return false;
        }
        
        return true;
    }

    /**
     * Get validation errors as array
     */
    protected function get_validation_errors() {
        return $this->form_validation->error_array();
    }

    /**
     * Handle reCAPTCHA verification (if implemented)
     */
    protected function verify_recaptcha($response) {
        // Implement reCAPTCHA verification here
        // For now, return true
        return true;
    }

    /**
     * Send contact email
     */
    protected function send_contact_email($data) {
        // Send confirmation to user
        $contact = (object) $data;
        send_contact_confirmation($contact);
        
        // Send notification to admin
        $site_name = $this->data['site_settings']['site_name'];
        $admin_email = $this->data['site_settings']['site_email'];
        
        $subject = "New Contact Message - $site_name";
        $message = "
        <h3>New Contact Message</h3>
        <p><strong>Name:</strong> {$data['name']}</p>
        <p><strong>Email:</strong> {$data['email']}</p>
        <p><strong>Phone:</strong> {$data['phone']}</p>
        <p><strong>Subject:</strong> {$data['subject']}</p>
        <p><strong>Message:</strong><br>" . nl2br($data['message']) . "</p>
        ";
        
        send_email($admin_email, $subject, $message);
    }

    /**
     * Handle file upload
     */
    /**
 * Handle file upload
 */
	protected function handle_upload($field_name, $type_or_path, $upload_path = null) {
		// Handle both calling conventions
		if ($upload_path === null) {
			// Called as: handle_upload('field', './uploads/path/', 'jpg|png')
			$config['upload_path'] = $type_or_path;
			$config['allowed_types'] = 'jpg|jpeg|png|gif';
		} else {
			// Called as: handle_upload('field', 'image', './uploads/path/')
			$config['upload_path'] = $upload_path;
			$config['allowed_types'] = ($type_or_path === 'image') ? 'jpg|jpeg|png|gif' : 'jpg|jpeg|png|gif|pdf|doc|docx';
		}
		
		$config['max_size'] = 5120; // 5MB
		$config['encrypt_name'] = TRUE;
		
		// Create directory if not exists
		if (!is_dir($config['upload_path'])) {
			mkdir($config['upload_path'], 0777, true);
		}
		
		$this->load->library('upload', $config);
		
		if ($this->upload->do_upload($field_name)) {
			return array(
				'status' => true,
				'data' => $this->upload->data()
			);
		} else {
			return array(
				'status' => false,
				'error' => $this->upload->display_errors('', '')
			);
		}
	}

    /**
     * Sanitize input data
     */
    protected function sanitize_input($data) {
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                $data[$key] = $this->sanitize_input($value);
            }
            return $data;
        }
        
        return clean_input($data);
    }

    /**
     * Check for bot/spam submission
     */
    protected function is_spam_submission($time_field = 'submit_time') {
        // Honeypot field check
        $honeypot = $this->input->post('website'); // Hidden field
        if (!empty($honeypot)) {
            return true;
        }
        
        // Time-based check (form submitted too quickly)
        $submit_time = $this->input->post($time_field);
        if ($submit_time && (time() - $submit_time) < 3) {
            return true;
        }
        
        return false;
    }

    /**
     * Generate form validation rules for contact form
     */
    protected function contact_form_rules() {
        return array(
            array(
                'field' => 'name',
                'label' => 'Name',
                'rules' => 'required|max_length[150]|regex_match[/^[a-zA-Z\s]+$/]'
            ),
            array(
                'field' => 'email',
                'label' => 'Email',
                'rules' => 'required|valid_email|max_length[150]'
            ),
            array(
                'field' => 'phone',
                'label' => 'Phone',
                'rules' => 'required|max_length[20]|regex_match[/^[0-9\s\+\-\(\)]+$/]'
            ),
            array(
                'field' => 'subject',
                'label' => 'Subject',
                'rules' => 'required|max_length[255]'
            ),
            array(
                'field' => 'message',
                'label' => 'Message',
                'rules' => 'required|min_length[10]'
            )
        );
    }

    /**
     * Generate form validation rules for consultation form
     */
    protected function consultation_form_rules() {
        return array(
            array(
                'field' => 'full_name',
                'label' => 'Full Name',
                'rules' => 'required|max_length[150]'
            ),
            array(
                'field' => 'email',
                'label' => 'Email',
                'rules' => 'required|valid_email|max_length[150]'
            ),
            array(
                'field' => 'phone',
                'label' => 'Phone',
                'rules' => 'required|max_length[20]'
            ),
            array(
                'field' => 'organization',
                'label' => 'Organization',
                'rules' => 'max_length[200]'
            ),
            array(
                'field' => 'consultation_type',
                'label' => 'Consultation Type',
                'rules' => 'required|in_list[individual,organization,community,other]'
            ),
            array(
                'field' => 'subject',
                'label' => 'Subject',
                'rules' => 'required|max_length[255]'
            ),
            array(
                'field' => 'message',
                'label' => 'Message',
                'rules' => 'required|min_length[20]'
            )
        );
    }

    /**
     * Generate form validation rules for volunteer form
     */
    protected function volunteer_form_rules() {
        return array(
            array(
                'field' => 'full_name',
                'label' => 'Full Name',
                'rules' => 'required|max_length[150]'
            ),
            array(
                'field' => 'email',
                'label' => 'Email',
                'rules' => 'required|valid_email|max_length[150]'
            ),
            array(
                'field' => 'phone',
                'label' => 'Phone',
                'rules' => 'required|max_length[20]'
            ),
            array(
                'field' => 'address',
                'label' => 'Address',
                'rules' => 'required'
            ),
            array(
                'field' => 'city',
                'label' => 'City',
                'rules' => 'required|max_length[100]'
            ),
            array(
                'field' => 'occupation',
                'label' => 'Occupation',
                'rules' => 'required|max_length[150]'
            ),
            array(
                'field' => 'skills',
                'label' => 'Skills',
                'rules' => 'required'
            ),
            array(
                'field' => 'availability',
                'label' => 'Availability',
                'rules' => 'required|in_list[weekdays,weekends,both,flexible]'
            ),
            array(
                'field' => 'motivation',
                'label' => 'Motivation',
                'rules' => 'required|min_length[50]'
            )
        );
    }
}