<?php
defined('BASEPATH') OR exit('No direct script access allowed');

if (!function_exists('array_to_csv')) {
    /**
     * Convert an array to CSV
     * 
     * @param array $array The array to convert
     * @param string $filename The name of the file to output
     * @param bool $headers Whether to output HTTP headers
     */
    function array_to_csv($array, $filename = 'export.csv', $headers = true) {
        // Ensure we have data to export
        if (empty($array)) {
            show_error('No data to export');
            return false;
        }

        // If headers are requested, send them
        if ($headers) {
            // Disable output buffering to prevent memory issues with large datasets
            if (ob_get_level()) {
                ob_end_clean();
            }

            // Send headers to force download
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            header('Pragma: no-cache');
            header('Expires: 0');
        }

        // Open output stream
        $fp = fopen('php://output', 'w');

        // If the array contains objects, convert to associative array
        $first_row = is_object(reset($array)) 
            ? (array) reset($array) 
            : reset($array);

        // Write the header row
        fputcsv($fp, array_keys($first_row));

        // Write data rows
        foreach ($array as $row) {
            // Convert to array if it's an object
            $row_array = is_object($row) ? (array) $row : $row;
            
            // Sanitize and convert to string
            $sanitized_row = array_map(function($value) {
                // Convert to string, handle null values
                if ($value === null) {
                    return '';
                }
                
                // If it's an array or object, json encode
                if (is_array($value) || is_object($value)) {
                    return json_encode($value);
                }
                
                // Convert to string
                return (string) $value;
            }, $row_array);

            // Write the row
            fputcsv($fp, $sanitized_row);
        }

        // Close the file handle
        fclose($fp);

        // Exit to prevent additional output
        exit();
    }
}

if (!function_exists('csv_to_array')) {
    /**
     * Convert a CSV file to an array
     * 
     * @param string $filename Path to the CSV file
     * @param bool $has_headers Whether the CSV has a header row
     * @return array Parsed CSV data
     */
    function csv_to_array($filename, $has_headers = true) {
        // Check if file exists
        if (!file_exists($filename)) {
            show_error('File not found: ' . $filename);
            return false;
        }

        // Open the CSV file
        $handle = fopen($filename, 'r');

        // Initialize array to store data
        $data = [];

        // Read headers if present
        $headers = [];
        if ($has_headers) {
            $headers = fgetcsv($handle);
        }

        // Read data rows
        while (($row = fgetcsv($handle)) !== FALSE) {
            // If headers were read, create an associative array
            if (!empty($headers)) {
                $data[] = array_combine($headers, $row);
            } else {
                $data[] = $row;
            }
        }

        // Close the file handle
        fclose($handle);

        return $data;
    }
}