<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Blog Model
 * Handles all blog-related database operations
 * Location: application/models/Blog_model.php
 */
class Blog_model extends CI_Model {

    private $table = 'blog_posts';
    private $category_table = 'blog_categories';

    public function __construct() {
        parent::__construct();
    }

    /**
     * Get all blog posts with pagination (admin)
     */
    public function get_posts($limit = 10, $offset = 0, $status = null) {
        $this->db->select('blog_posts.*, blog_categories.name as category_name, 
                          blog_categories.slug as category_slug, 
                          admin_users.full_name as author_name');
        $this->db->from($this->table);
        $this->db->join('blog_categories', 'blog_categories.id = blog_posts.category_id', 'left');
        $this->db->join('admin_users', 'admin_users.id = blog_posts.author_id', 'left');
        
        // Only filter by status if provided
        if ($status !== null && $status !== '') {
            $this->db->where('blog_posts.status', $status);
        }
        
        // Order by created_at for better sorting (works for all statuses)
        $this->db->order_by('blog_posts.created_at', 'DESC');
        $this->db->limit($limit, $offset);
        
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Get total count of posts (with optional status filter)
     */
    public function count_posts($status = null, $category_id = null) {
        $this->db->from($this->table);
        
        // Only filter by status if provided
        if ($status !== null && $status !== '') {
            $this->db->where('status', $status);
        }
        
        if ($category_id) {
            $this->db->where('category_id', $category_id);
        }
        
        return $this->db->count_all_results();
    }

    /**
     * Get single post by slug
     */
    public function get_post_by_slug($slug) {
        $this->db->select('blog_posts.*, blog_categories.name as category_name, 
                          blog_categories.slug as category_slug, 
                          admin_users.full_name as author_name,
                          admin_users.email as author_email');
        $this->db->from($this->table);
        $this->db->join('blog_categories', 'blog_categories.id = blog_posts.category_id', 'left');
        $this->db->join('admin_users', 'admin_users.id = blog_posts.author_id', 'left');
        $this->db->where('blog_posts.slug', $slug);
        $this->db->where('blog_posts.status', 'published');
        
        $query = $this->db->get();
        return $query->row();
    }

    /**
     * Get post by ID (for admin)
     */
    public function get_post_by_id($id) {
        $this->db->select('blog_posts.*, blog_categories.name as category_name');
        $this->db->from($this->table);
        $this->db->join('blog_categories', 'blog_categories.id = blog_posts.category_id', 'left');
        $this->db->where('blog_posts.id', $id);
        
        $query = $this->db->get();
        return $query->row();
    }

    /**
     * Get posts by category
     */
    public function get_posts_by_category($category_slug, $limit = 10, $offset = 0) {
        $this->db->select('blog_posts.*, blog_categories.name as category_name, 
                  blog_categories.slug as category_slug, 
                  admin_users.full_name as author_name,
                  blog_posts.views_count as views'); 
        $this->db->from($this->table);
        $this->db->join('blog_categories', 'blog_categories.id = blog_posts.category_id', 'left');
        $this->db->join('admin_users', 'admin_users.id = blog_posts.author_id', 'left');
        $this->db->where('blog_categories.slug', $category_slug);
        $this->db->where('blog_posts.status', 'published');
        $this->db->order_by('blog_posts.published_at', 'DESC');
        $this->db->limit($limit, $offset);
        
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Get recent posts
     */
    public function get_recent_posts($limit = 5) {
        $this->db->select('id, title, slug, featured_image, published_at, excerpt');
        $this->db->from($this->table);
        $this->db->where('status', 'published');
        $this->db->order_by('published_at', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Get related posts based on category
     */
    public function get_related_posts($category_id, $current_post_id, $limit = 3) {
        $this->db->select('id, title, slug, featured_image, published_at, excerpt');
        $this->db->from($this->table);
        $this->db->where('category_id', $category_id);
        $this->db->where('id !=', $current_post_id);
        $this->db->where('status', 'published');
        $this->db->order_by('published_at', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Increment view count
     */
    public function increment_views($id) {
        $this->db->set('views_count', 'views_count + 1', FALSE);
        $this->db->where('id', $id);
        return $this->db->update($this->table);
    }

    /**
     * Create new blog post
     */
    public function create_post($data) {
        // Generate slug if not provided
        if (empty($data['slug'])) {
            $data['slug'] = $this->generate_slug($data['title']);
        }
        
        // Set published_at if status is published
        if ($data['status'] == 'published' && empty($data['published_at'])) {
            $data['published_at'] = date('Y-m-d H:i:s');
        }
        
        return $this->db->insert($this->table, $data);
    }

    /**
     * Update blog post
     */
    public function update_post($id, $data) {
        // Generate new slug if title changed
        if (isset($data['title']) && empty($data['slug'])) {
            $data['slug'] = $this->generate_slug($data['title'], $id);
        }
        
        // Set published_at if status changed to published
        if (isset($data['status']) && $data['status'] == 'published') {
            $existing = $this->get_post_by_id($id);
            if (empty($existing->published_at)) {
                $data['published_at'] = date('Y-m-d H:i:s');
            }
        }
        
        $this->db->where('id', $id);
        return $this->db->update($this->table, $data);
    }

    /**
     * Delete blog post
     */
    public function delete_post($id) {
        $this->db->where('id', $id);
        return $this->db->delete($this->table);
    }

    /**
     * Generate unique slug
     */
    private function generate_slug($title, $exclude_id = null) {
        $this->load->helper('text');
        $slug = url_title(convert_accented_characters($title), 'dash', TRUE);
        
        // Check if slug exists
        $this->db->where('slug', $slug);
        if ($exclude_id) {
            $this->db->where('id !=', $exclude_id);
        }
        $count = $this->db->count_all_results($this->table);
        
        // Append number if slug exists
        if ($count > 0) {
            $slug = $slug . '-' . ($count + 1);
        }
        
        return $slug;
    }

    // ========== CATEGORY METHODS ==========

    /**
     * Get all active categories
     */
    public function get_categories($status = 'active') {
        $this->db->where('status', $status);
        $this->db->order_by('name', 'ASC');
        $query = $this->db->get($this->category_table);
        return $query->result();
    }

    /**
     * Get category by slug
     */
    public function get_category_by_slug($slug) {
        $this->db->where('slug', $slug);
        $query = $this->db->get($this->category_table);
        return $query->row();
    }

    /**
     * Get category by ID
     */
    public function get_category_by_id($id) {
        $this->db->where('id', $id);
        $query = $this->db->get($this->category_table);
        return $query->row();
    }

    /**
     * Create category
     */
    public function create_category($data) {
        if (empty($data['slug'])) {
            $data['slug'] = url_title($data['name'], 'dash', TRUE);
        }
        return $this->db->insert($this->category_table, $data);
    }

    /**
     * Update category
     */
    public function update_category($id, $data) {
        if (isset($data['name']) && empty($data['slug'])) {
            $data['slug'] = url_title($data['name'], 'dash', TRUE);
        }
        $this->db->where('id', $id);
        return $this->db->update($this->category_table, $data);
    }

    /**
     * Delete category
     */
    public function delete_category($id) {
        $this->db->where('id', $id);
        return $this->db->delete($this->category_table);
    }

    /**
     * Get categories with post count
     */
    public function get_categories_with_count() {
        $this->db->select('blog_categories.*, COUNT(blog_posts.id) as post_count');
        $this->db->from($this->category_table);
        $this->db->join('blog_posts', 'blog_posts.category_id = blog_categories.id AND blog_posts.status = "published"', 'left');
        $this->db->where('blog_categories.status', 'active');
        $this->db->group_by('blog_categories.id');
        $this->db->order_by('blog_categories.name', 'ASC');
        
        $query = $this->db->get();
        return $query->result();
    }
}