<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Common Model
 * Handles workshops, partners, carousel, and site settings
 * Location: application/models/Common_model.php
 */
class Common_model extends CI_Model {

    public function __construct() {
        parent::__construct();
    }

    // ========== WORKSHOP METHODS ==========

    /**
     * Get all workshops
     */
    public function get_workshops($status = null, $limit = null) {
        $this->db->select('*');
        $this->db->from('workshop_programmes');
        
        if ($status) {
            $this->db->where('status', $status);
        }
        
        $this->db->order_by('display_order', 'ASC');
        $this->db->order_by('start_date', 'DESC');
        
        if ($limit) {
            $this->db->limit($limit);
        }
        
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Get workshop by slug
     */
    public function get_workshop_by_slug($slug) {
        $this->db->where('slug', $slug);
        $query = $this->db->get('workshop_programmes');
        return $query->row();
    }

    /**
     * Get workshop by ID
     */
    public function get_workshop($id) {
        $this->db->where('id', $id);
        $query = $this->db->get('workshop_programmes');
        return $query->row();
    }

    /**
     * Get featured workshops
     */
    public function get_featured_workshops($limit = 3) {
        $this->db->where('is_featured', 1);
        $this->db->where('status', 'upcoming');
        $this->db->order_by('start_date', 'ASC');
        $this->db->limit($limit);
        $query = $this->db->get('workshop_programmes');
        return $query->result();
    }

    /**
     * Create workshop
     */
    public function create_workshop($data) {
        if (empty($data['slug'])) {
            $data['slug'] = url_title($data['title'], 'dash', TRUE);
        }
        return $this->db->insert('workshop_programmes', $data);
    }

    /**
     * Update workshop
     */
    public function update_workshop($id, $data) {
        if (isset($data['title']) && empty($data['slug'])) {
            $data['slug'] = url_title($data['title'], 'dash', TRUE);
        }
        $this->db->where('id', $id);
        return $this->db->update('workshop_programmes', $data);
    }

    /**
     * Delete workshop
     */
    public function delete_workshop($id) {
        $this->db->where('id', $id);
        return $this->db->delete('workshop_programmes');
    }

    // ========== PARTNER METHODS ==========

    /**
     * Get all partners
     */
    public function get_partners($status = 'active') {
        $this->db->where('status', $status);
        $this->db->order_by('display_order', 'ASC');
        $this->db->order_by('name', 'ASC');
        $query = $this->db->get('partners');
        return $query->result();
    }

    /**
     * Get partner by ID
     */
    public function get_partner($id) {
        $this->db->where('id', $id);
        $query = $this->db->get('partners');
        return $query->row();
    }

    /**
     * Create partner
     */
    public function create_partner($data) {
        return $this->db->insert('partners', $data);
    }

    /**
     * Update partner
     */
    public function update_partner($id, $data) {
        $this->db->where('id', $id);
        return $this->db->update('partners', $data);
    }

    /**
     * Delete partner
     */
    public function delete_partner($id) {
        $this->db->where('id', $id);
        return $this->db->delete('partners');
    }

    // ========== CAROUSEL METHODS ==========

    /**
     * Get all carousel items
     */
    public function get_carousel_items($status = 'active') {
        $this->db->where('status', $status);
        $this->db->order_by('display_order', 'ASC');
        $query = $this->db->get('home_carousel');
        return $query->result();
    }

    /**
     * Get carousel item by ID
     */
    public function get_carousel_item($id) {
        $this->db->where('id', $id);
        $query = $this->db->get('home_carousel');
        return $query->row();
    }

    /**
     * Create carousel item
     */
    public function create_carousel_item($data) {
        return $this->db->insert('home_carousel', $data);
    }

    /**
     * Update carousel item
     */
    public function update_carousel_item($id, $data) {
        $this->db->where('id', $id);
        return $this->db->update('home_carousel', $data);
    }

    /**
     * Delete carousel item
     */
    public function delete_carousel_item($id) {
        $this->db->where('id', $id);
        return $this->db->delete('home_carousel');
    }

    // ========== SITE SETTINGS METHODS ==========

    /**
     * Get all settings
     */
    public function get_all_settings() {
        $query = $this->db->get('site_settings');
        $settings = array();
        
        foreach ($query->result() as $row) {
            $settings[$row->setting_key] = $row->setting_value;
        }
        
        return $settings;
    }

    /**
     * Get settings by group
     */
    public function get_settings_by_group($group) {
        $this->db->where('setting_group', $group);
        $query = $this->db->get('site_settings');
        
        $settings = array();
        foreach ($query->result() as $row) {
            $settings[$row->setting_key] = $row->setting_value;
        }
        
        return $settings;
    }

    /**
     * Get single setting
     */
    public function get_setting($key) {
        $this->db->where('setting_key', $key);
        $query = $this->db->get('site_settings');
        $result = $query->row();
        
        return $result ? $result->setting_value : null;
    }

    /**
     * Update setting
     */
    public function update_setting($key, $value) {
        $this->db->where('setting_key', $key);
        $exists = $this->db->get('site_settings')->num_rows();
        
        if ($exists) {
            $this->db->where('setting_key', $key);
            return $this->db->update('site_settings', array('setting_value' => $value));
        } else {
            return $this->db->insert('site_settings', array(
                'setting_key' => $key,
                'setting_value' => $value
            ));
        }
    }

    /**
     * Update multiple settings
     */
    public function update_settings($settings) {
        $this->db->trans_start();
        
        foreach ($settings as $key => $value) {
            $this->update_setting($key, $value);
        }
        
        $this->db->trans_complete();
        return $this->db->trans_status();
    }

    // ========== EMAIL LOG METHODS ==========

    /**
     * Log email
     */
    public function log_email($data) {
        return $this->db->insert('email_logs', $data);
    }

    /**
     * Get email logs
     */
    public function get_email_logs($limit = 50, $offset = 0) {
        $this->db->order_by('sent_at', 'DESC');
        $this->db->limit($limit, $offset);
        $query = $this->db->get('email_logs');
        return $query->result();
    }

    // ========== ADMIN USER METHODS ==========

    /**
     * Get admin by username
     */
    public function get_admin_by_username($username) {
        $this->db->where('username', $username);
        $this->db->where('status', 'active');
        $query = $this->db->get('admin_users');
        return $query->row();
    }

    /**
     * Get admin by email
     */
    public function get_admin_by_email($email) {
        $this->db->where('email', $email);
        $this->db->where('status', 'active');
        $query = $this->db->get('admin_users');
        return $query->row();
    }

    /**
     * Get admin by ID
     */
    public function get_admin($id) {
        $this->db->where('id', $id);
        $query = $this->db->get('admin_users');
        return $query->row();
    }

    /**
     * Update admin profile
     */
    public function update_admin($id, $data) {
        $this->db->where('id', $id);
        return $this->db->update('admin_users', $data);
    }

    /**
     * Update admin password
     */
    public function update_admin_password($id, $password) {
        $data = array('password' => password_hash($password, PASSWORD_DEFAULT));
        $this->db->where('id', $id);
        return $this->db->update('admin_users', $data);
    }

    /**
     * Update last login
     */
    public function update_last_login($id) {
        $data = array('last_login' => date('Y-m-d H:i:s'));
        $this->db->where('id', $id);
        return $this->db->update('admin_users', $data);
    }

    /**
     * Get all admins
     */
    public function get_all_admins() {
        $this->db->select('id, username, email, full_name, role');
        $this->db->where('status', 'active');
        $this->db->order_by('full_name', 'ASC');
        $query = $this->db->get('admin_users');
        return $query->result();
    }
	/**
 * Count workshops by status
 */
public function count_workshops($status = null) {
    $this->db->from('workshop_programmes'); // <-- Change to your actual table name
    
    if ($status !== null && $status !== '') {
        $this->db->where('status', $status);
    }
    
    return $this->db->count_all_results();
}	
}