<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Kid_registration_model extends CI_Model {
    private $table = 'french_kid_learners';

    public function __construct() {
        parent::__construct();
    }

    /**
     * Register a new kid learner
     * @param array $data Registration data
     * @return bool|int
     */
    public function register($data) {
        // Start a database transaction
        $this->db->trans_start();

        // Insert registration data
        $this->db->insert($this->table, $data);
        $insert_id = $this->db->insert_id();

        // Log the registration in audit log
        $this->_log_registration($insert_id, 'Created');

        // Complete the transaction
        $this->db->trans_complete();

        // Return the result
        return $this->db->trans_status() ? $insert_id : FALSE;
    }

    /**
     * Get all registrations
     * @param string $status Optional status filter
     * @return array
     */
    public function get_all_registrations($status = NULL) {
        if ($status) {
            $this->db->where('status', $status);
        }
        return $this->db->order_by('created_at', 'DESC')
                        ->get($this->table)
                        ->result();
    }

    /**
     * Get a specific registration by ID
     * @param int $id Registration ID
     * @return object|null
     */
    public function get_registration($id) {
        return $this->db->where('id', $id)
                        ->get($this->table)
                        ->row();
    }

    /**
     * Update registration status
     * @param int $id Registration ID
     * @param string $status New status
     * @return bool
     */
    public function update_status($id, $status) {
        // Start a database transaction
        $this->db->trans_start();

        // Get the current status before update
        $current_registration = $this->get_registration($id);
        $old_status = $current_registration ? $current_registration->status : NULL;

        // Update the status
        $update_data = [
            'status' => $status,
            'updated_by' => $this->session->userdata('user_id') // Assuming admin is logged in
        ];
        $this->db->where('id', $id)
                 ->update($this->table, $update_data);

        // Log the status change in audit log
        $this->_log_status_change($id, $old_status, $status);

        // Complete the transaction
        $this->db->trans_complete();

        // Return the result
        return $this->db->trans_status();
    }

    /**
     * Search registrations
     * @param array $filters Search filters
     * @return array
     */
    public function search($filters = []) {
        // Apply dynamic filters
        if (!empty($filters['name'])) {
            $this->db->group_start()
                     ->like('first_name', $filters['name'])
                     ->or_like('surname', $filters['name'])
                     ->group_end();
        }

        if (!empty($filters['parent_name'])) {
            $this->db->like('parent_name', $filters['parent_name']);
        }

        if (!empty($filters['email'])) {
            $this->db->like('email', $filters['email']);
        }

        if (!empty($filters['french_level'])) {
            $this->db->where('french_level', $filters['french_level']);
        }

        if (!empty($filters['status'])) {
            $this->db->where('status', $filters['status']);
        }

        return $this->db->get($this->table)->result();
    }

    /**
     * Log registration creation
     * @param int $registration_id Registration ID
     * @param string $action Action type
     */
    private function _log_registration($registration_id, $action) {
        $log_data = [
            'registration_type' => 'Kid',
            'registration_id' => $registration_id,
            'action' => $action,
            'performed_by' => $this->session->userdata('user_id') // Optional
        ];

        $this->db->insert('french_registration_audit_log', $log_data);
    }

    /**
     * Log status changes
     * @param int $registration_id Registration ID
     * @param string $old_status Previous status
     * @param string $new_status New status
     */
    private function _log_status_change($registration_id, $old_status, $new_status) {
        $log_data = [
            'registration_type' => 'Kid',
            'registration_id' => $registration_id,
            'action' => 'Status Changed',
            'old_status' => $old_status,
            'new_status' => $new_status,
            'performed_by' => $this->session->userdata('user_id') // Optional
        ];

        $this->db->insert('french_registration_audit_log', $log_data);
    }
}